 /**
* \file: FeatureDiscoveryTypes.h
*
* \version: 0.1
*
* \release: $Name:$
*
* Definitions of structures which are used inside the Feature Discovery.
*
* \component: Unified SPI
*
* \author: D. Girnus / ADIT/SW2 / dgirnus@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef USPI_FEATURE_DISCOVERY_STRUCTS_H_
#define USPI_FEATURE_DISCOVERY_STRUCTS_H_


#include <stdint.h>
#include <string>
#include <memory>

/* included to use class DeviceInfo to access device information received from Device Detector */
#include <uspi/deviceManagement.h>
/* included to know the device mask that has been triggered by Device Detector */
#include <uspi/deviceDetector.h>


namespace adit { namespace uspi {


/* forward declaration of Transport classes */
template<typename Type>
class Transport;

class UsbTransport;
class BtTransport;
class WiFiTransport;


/**
 * This class represents to possible return and error values
 * of the FeatureDiscovery.
 *
 */
enum class DiscoveryError : int32_t
{
    /* common error values */
    OK          = 0,            /**< Operation succeed. */
    FAILURE,                    /**< Operation failed. */
    INVALID,                    /**< Operation or parameter is invalid. */
    NOMEM,                      /**< Out of memory. */
    /* transport related error values */
    NO_DEVICE,                  /**< No device (maybe disconnected). */
    NOT_FOUND,                  /**< Entity not found. */
    BUSY,                       /**< Operation already in progress or resource busy. */
    TIMEOUT,                    /**< Operation timed out. */
    TRANSFER_OVERFLOW,          /**< Overflow. */
    PIPE,                       /**< Pipe error. */
    IO,                         /**< Input/output error. */
    /* additional error values */
    INCOMPLETE,                 /**< Operation not complete (e.g. resetDevice() not done) */
    UNSUPPORTED,                /**< Operation not supported. */
    MAX                         /**< Maximum enumeration value */
};

static const char* DiscoveryErrorStrings[] = { "OK", "FAILURE", "INVALID", "NOMEM", "NO_DEVICE", "NOT_FOUND", \
                                               "BUSY", "TIMEOUT", "TRANSFER_OVERFLOW", "PIPE", "IO", "INCOMPLETE", \
                                               "UNSUPPORTED", "MAX" };
inline const char* to_str(const DiscoveryError error)
{
    return DiscoveryErrorStrings[static_cast<int32_t>(error)];
}

/**
 * \brief Supported FeatureDiscovery protocol types.
 *        Lists the different kind of protocols that will be supported by the FeatureDiscovery.
 */
enum SpiFeatureProtocol
{
    FD_PROTOCOL_NONE                    = 0,
    FD_PROTOCOL_GOOGLE_AOAP             = 1,    // could support AndroidAuto, MySpin, Baidu CarLife (only determine after switch was done)
    FD_PROTOCOL_APPLE_CARPLAY           = 2,    // support Apple CarPlay
    FD_PROTOCOL_APPLE_NATIVE_HOST_MODE  = 3,    // support App communication via Native Host Mode - MySpin, Baidu CarLife
    FD_PROTOCOL_MIRRORLINK              = 4     // support MirrorLink (only determine after switch was done)
};

/**
 * \brief Structure of the required string information to switch to AOAP mode.
 */
struct SpiAoapInformation
{
    std::string manufacturer; // The manufacturer of the MD application (for identification of Android application)
    std::string modelName;    // The model name of the MD application (for identification of Android application)
    std::string description;  // The description of the MD application (for identification of Android application)
    std::string version;      // The version (for identification of Android application)
    std::string uri;          // The URI where to download the MD application (for identification of Android application)
    std::string serial;       // The serial number of the HU
    uint32_t    enableAudio; // =0 : disable audio. >0 : enable audio (only necessary for AOA 2.0)
};


/**
 * \brief The base class of the discovered device classes.
 *
 * It is used as generic type to inform upper layer via IFeatureDiscoveryCb
 * about device connections/disconnections.
 */
class DiscoveredDevice
{
public:
    DiscoveredDevice() : mEventMask(0) {}
    virtual ~DiscoveredDevice() {}

    virtual void* getInfo() { return nullptr; }

    virtual uint32_t getEventMask(void) const { return mEventMask; }

private:

    /* The device mask that has been triggered.
     * Refer to enum DD_EVENT_MASK in deviceDetector.h */
    uint32_t mEventMask;
};

/**
 * \brief The USB device class.
 *
 * This class is meant to be derived by DiscoveredDevice.
 * It provides functionalities to check if the device supports a specific SPI feature,
 * to switch the device to one of the SPI feature and to access the device information.
 */
class DiscoveredDeviceUsb : public DiscoveredDevice
{
public:
    DiscoveredDeviceUsb(const DeviceInfo& inDevInfo, uint32_t inEventMask);
    virtual ~DiscoveredDeviceUsb();


    /* API's to communicate with the mobile device */

    /**
     * \brief Checks if the device supports the SPI Feature Protocol.
     *
     * \param inProtocol The SPI feature protocol which should be checked (AOAP, CarPlay)
     * \param inTimeoutMs Indicates that the API should be blocking (-1) or not (value >0).
     * \return true if protocol is supported (e.g. in case of AOAP: AAUTO, MYSPIN, etc.), otherwise false
     *
     */
    bool checkDevice(SpiFeatureProtocol inProtocol, int32_t inTimeoutMs);

    /**
     * \brief  Switch the mobile device into the requested mode to support the SPI feature.
     *
     * \param inProtocol The SPI feature protocol which should be checked (AOAP, CarPlay)
     * \param inParam Additional parameters that may be send before switch command
     *                to prepare mobile device (e.g. string information in case of AOAP)
     * \param inTimeoutMs Indicates that the API should be blocking (-1) or not (value >0).
     *
     * \return 0 in case of success. Otherwise negative return value.
     */
    DiscoveryError switchDevice(SpiFeatureProtocol inProtocol, void* inParam, int32_t inTimeoutMs);

    /**
     * \brief  Try to bring the mobile device out of the current switched mode into normal mode.
     *
     * \param inProtocol The SPI feature protocol which should be checked (AOAP, CarPlay)
     * \param inTimeoutMs Indicates that the API should be blocking (-1) or not (value >0).
     *
     * \return 0 in case of success. Otherwise negative return value.
     */
    DiscoveryError resetDevice(SpiFeatureProtocol inProtocol, int32_t inTimeoutMs);

    /**
     * \brief  Returns the Device information received from Device Detector.
     *
     * \return Device information of type DeviceInfo
     */
    DeviceInfo getInfo(void) const { return mInfo; }

    /**
     * \brief  Returns the event mask received from Device Detector.
     *
     * \return Information of the connection type like USB, BT or WiFi
     */
    uint32_t getEventMask(void) const { return (uint32_t)mInfo.mMask; }
private:
    /**
     * \brief  Internal helper API to send the AOAP string information.
     *
     * \param inParam Additional parameters that may be send before switch command
     *                to prepare mobile device (e.g. string information in case of AOAP)
     * \param inTimeoutMs Indicates that the API should be blocking (-1) or not (value >0).
     *
     * \return 0 in case of success. Otherwise negative return value.
     */
    DiscoveryError sendStringInformation(void* inParam, int32_t inTimeoutMs);

    /* Device information received from Device Detector */
    DeviceInfo mInfo;

    /* Transport which reference to the specific transport layer */
    Transport<UsbTransport>* mTransport;

    /* The device mask that has been triggered.
     * Refer to enum DD_EVENT_MASK in deviceDetector.h */
    uint32_t mEventMask;
};

/**
 * \brief The BT device object class.
 *
 * This class is meant to be derived by FdBaseObject.
 * It provides functionalities to check if the device supports a specific SPI feature,
 * to switch the device to one of the SPI feature and to access the device information.
 */
class DiscoveredDeviceBt : public DiscoveredDevice
{
public:

    DiscoveredDeviceBt(const DeviceInfo& inDevInfo, uint32_t inEventMask) {
        mInfo = inDevInfo;
        mTransport = nullptr;
        mEventMask = inEventMask;
    }
    virtual ~DiscoveredDeviceBt() {}

    /**
     * \brief  Returns the Device information received from Device Detector.
     *
     * \return Device information of type DeviceInfo
     */
    DeviceInfo getInfo(void) const { return mInfo; }

    /**
     * \brief  Returns the event mask received from Device Detector.
     *
     * \return Information of the connection type like USB, BT or WiFi
     */
    uint32_t getEventMask(void) const { return (uint32_t)mInfo.mMask; }

private:
    /* Device information received from Device Detector */
    DeviceInfo mInfo;

    /* Transport which reference to the specific transport layer */
    Transport<BtTransport>* mTransport;

    /* The device mask that has been triggered.
     * Refer to enum DD_EVENT_MASK in deviceDetector.h */
    uint32_t mEventMask;
};

/**
 * \brief The WiFi device object class.
 *
 * This class is meant to be derived by FdBaseObject.
 * It provides functionalities to check if the device supports a specific SPI feature,
 * to switch the device to one of the SPI feature and to access the device information.
 */
class DiscoveredDeviceWiFi : public DiscoveredDevice
{
public:

    DiscoveredDeviceWiFi(const DeviceInfo& inDevInfo, uint32_t inEventMask) {
        mInfo = inDevInfo;
        mTransport = nullptr;
        mEventMask = inEventMask;
    }
    virtual ~DiscoveredDeviceWiFi() {}

    /**
     * \brief  Returns the Device information received from Device Detector.
     *
     * \return Device information of type DeviceInfo
     */
    DeviceInfo getInfo(void) const { return mInfo; }

    /**
     * \brief  Returns the event mask received from Device Detector.
     *
     * \return Information of the connection type like USB, BT or WiFi
     */
    uint32_t getEventMask(void) const { return (uint32_t)mInfo.mMask; }
private:
    /* Device information received from Device Detector */
    DeviceInfo mInfo;

    /* Transport which reference to the specific transport layer */
    Transport<WiFiTransport>* mTransport;

    /* The device mask that has been triggered.
     * Refer to enum DD_EVENT_MASK in deviceDetector.h */
    uint32_t mEventMask;
};


} } /* namespace adit { namespace uspi { */


#endif /* USPI_FEATURE_DISCOVERY_STRUCTS_H_ */
